#
# ScopedGUI.py
#

from Tkinter import *
from Tkinter import PhotoImage
import tkMessageBox

from DCharts import *
from ScopedStatechart import ScopedStatechart

CANVAS_W = 900
CANVAS_H = 900

# keep track of button coordinates
statX0 = 10
statY0 = 10
statX1 = 115
statY1 = 90

orthX0 = 135
orthY0 = 10
orthX1 = 240
orthY1 = 90

compX0 = 260
compY0 = 10
compX1 = 365
compY1 = 90

histX0 = 385
histY0 = 10
histX1 = 490
histY1 = 90
            
class ScopedGUI:

    def __init__(self, parent):     
      self.controller = ScopedGUI_Controller()    
      self.staticGUI  = ScopedGUI_Static(parent, self.controller)
      self.dynamicGUI = ScopedGUI_Dynamic(self.controller)
   
        
class ScopedGUI_Controller:
    def __init__(self):
      self.statechart = None
      self.GUI  = None

    def bindDynamic(self, statechart):
      self.statechart = statechart
  
    def bindStatic(self, GUI):
      self.GUI = GUI

    #########################
    # Interface for the GUI #
    #########################
                             
    def window_close(self):
      import sys
      sys.exit(0)
      self.statechart.event('GUI Quit')

    # these are all the triggers we send to the statechart

    def Reset(self):
      self.statechart.event('Reset')

    def CompositeButton(self):
      self.statechart.event('CompositeButton')

    def OrthogonalButton(self):
      self.statechart.event('OrthogonalButton')

    def StateButton(self):
      self.statechart.event('StateButton')

    def HistoryButton(self):
      self.statechart.event('HistoryButton')

    def Create(self):
      self.statechart.event('Create')

    def ButtonPress1(self):
      self.statechart.event('ButtonPress1')

    def ButtonRelease1(self):
      self.statechart.event('ButtonRelease1')

    def ButtonPress3(self):
      self.statechart.event('ButtonPress3')

    def ControlButtonPress1(self):
      self.statechart.event('ControlButtonPress1')

    def ControlButtonPress3(self):
      self.statechart.event('ControlButtonPress3')

    def Done(self):
      self.statechart.event('Done')

    def AnyMotion(self):
      self.statechart.event('AnyMotion')

    def GiveOptions(self):
      self.statechart.event('GiveOptions')

    def Select(self):
      self.statechart.event('Select')

    def Deselect(self):
      self.statechart.event('Deselect')

    def BasicSelect(self):
      self.statechart.event('BasicSelect')

    def OrthogonalSelect(self):
      self.statechart.event('OrthogonalSelect')

    def CompositeSelect(self):
      self.statechart.event('CompositeSelect')

    def HistorySelect(self):
      self.statechart.event('HistorySelect')

    def EdgeSelect(self):
      self.statechart.event('EdgeSelect')

    def BasicDeselect(self):
      self.statechart.event('BasicDeselect')

    def OrthogonalDeselect(self):
      self.statechart.event('OrthogonalDeselect')

    def CompositeDeselect(self):
      self.statechart.event('CompositeDeselect')

    def HistoryDeselect(self):
      self.statechart.event('HistoryDeselect')

    def EdgeDeselect(self):
      self.statechart.event('EdgeDeselect')

    def Move(self):
      self.statechart.event('Move')

    def BasicMove(self):
      self.statechart.event('BasicMove')

    def OrthogonalMove(self):
      self.statechart.event('OrthogonalMove')

    def CompositeMove(self):
      self.statechart.event('CompositeMove')

    def HistoryMove(self):
      self.statechart.event('HistoryMove')

    def EdgeMove(self):
      self.statechart.event('EdgeMove')

    def Contain(self):
      self.statechart.event('Contain')

    def ContainOrthogonal(self):
      self.statechart.event('ContainOrthogonal')

    def ContainComposite(self):
      self.statechart.event('ContainComposite')

    def Delete(self):
      self.statechart.event('Delete')

    def BasicDelete(self):
      self.statechart.event('BasicDelete')

    def OrthogonalDelete(self):
      self.statechart.event('OrthogonalDelete')

    def CompositeDelete(self):
      self.statechart.event('CompositeDelete')

    def HistoryDelete(self):
      self.statechart.event('HistoryDelete')

    def EdgeDelete(self):
      self.statechart.event('EdgeDelete')


    ################################
    # Interface for the statechart #
    ################################

    # these are all the commands we get from the statechart

    # need to draw a composite state
    def CreateComposite(self):
      self.GUI.drawCompositeState()
      self.Done()

    # need to draw an orthogonal state
    def CreateOrthogonal(self):
      self.GUI.drawOrthogonalState()
      self.Done()

    # need to draw a basic state
    def CreateBasic(self):
      self.GUI.drawBasicState()
      self.Done()

    # need to draw a history state
    def CreateHistory(self):
      self.GUI.drawHistoryState()
      self.Done()

    # need to send a Create event
    def CreateEntity(self):
      self.Create()

    # need to draw a showLine
    def DrawShowLine(self):
      self.GUI.drawShowLine()

    # need to create an edge
    def CreateEdge(self):
      result = self.GUI.createEdgeHelper()
      if result == "transition":
        self.GUI.drawTransition()
        self.GUI.arrowLock = False
        self.Done()
      elif result == "contain":
        self.GUI.drawContainment()
        self.GUI.arrowLock = False
        self.Done()
        self.Contain() # need to send contain event
      elif result == "failed":
        self.GUI.arrowLock = False
        self.Done()
      elif result == "giveoptions":
        self.GiveOptions()

    # need to give options to the user for the creation of an edge coming from a composite state
    def DrawEdgeOptions(self):
      result = self.GUI.drawEdgeOptions()
      if result == "transition":
        self.GUI.drawTransition()
        self.GUI.arrowLock = False
        self.Done()
      elif result == "contain":
        self.GUI.drawContainment()
        self.GUI.arrowLock = False
        self.Done()
        self.Contain() # need to send contain event

    # need to deselect an entity
    def DeselectEntity(self):
      result = self.GUI.typeHelper(self.GUI.toDeselect)
      # send event to draw deselection of previous selection
      if result == "basic":
        self.BasicDeselect()
      elif result == "orthogonal":
        self.OrthogonalDeselect()
      elif result == "composite":
        self.CompositeDeselect()
      elif result == "history":
        self.HistoryDeselect()
      elif result == "edge":
        self.EdgeDeselect()


    # need to select an entity
    def SelectEntity(self):
      result = self.GUI.typeHelper(self.GUI.selected)
      # send event to draw new selection
      if result == "basic":
        self.BasicSelect()
      elif result == "orthogonal":
        self.OrthogonalSelect()
      elif result == "composite":
        self.CompositeSelect()
      elif result == "history":
        self.HistorySelect()
      elif result == "edge":
        self.EdgeSelect()

    # need to draw selection of a basic state
    def SelectBasic(self):
      self.GUI.drawBasicSelect(self.GUI.selected)
    
    # need to draw selection of an orthogonal state
    def SelectOrthogonal(self):
      outedgelist = self.GUI.outEdgeListHelper(self.GUI.selected[0])
      self.GUI.drawOrthogonalSelect(self.GUI.selected, outedgelist)

    # need to draw selection of a composite state
    def SelectComposite(self):
      outedgelist = self.GUI.outEdgeListHelper(self.GUI.selected[0])
      self.GUI.drawCompositeSelect(self.GUI.selected, outedgelist)

    # need to draw selection of a history state
    def SelectHistory(self):
      self.GUI.drawHistorySelect(self.GUI.selected)

    # need to draw selection of an edge
    def SelectEdge(self):
      self.GUI.drawEdgeSelect(self.GUI.selected)

    # need to draw deselection of a basic state
    def DeselectBasic(self):
      self.GUI.drawBasicDeselect(self.GUI.toDeselect)
    
    # need to draw deselection of an orthogonal state
    def DeselectOrthogonal(self):
      outedgelist = self.GUI.outEdgeListHelper(self.GUI.toDeselect[0])
      self.GUI.drawOrthogonalDeselect(self.GUI.toDeselect, outedgelist)

    # need to draw deselection of a composite state
    def DeselectComposite(self):
      outedgelist = self.GUI.outEdgeListHelper(self.GUI.toDeselect[0])
      self.GUI.drawCompositeDeselect(self.GUI.toDeselect, outedgelist)

    # need to draw deselection of a history state
    def DeselectHistory(self):
      self.GUI.drawHistoryDeselect(self.GUI.toDeselect)

    # need to draw deselection of an edge
    def DeselectEdge(self):
      self.GUI.drawEdgeDeselect(self.GUI.toDeselect)

    # need to move an entity
    def MoveEntity(self):
      result = self.GUI.typeHelper(self.GUI.selected)
      # send event to move entity
      if result == "basic":
        self.BasicMove()
      elif result == "orthogonal":
        self.OrthogonalMove()
      elif result == "composite":
        self.CompositeMove()
      elif result == "history":
        self.HistoryMove()

      # move the edges which are associated with the entity we have to move
      self.EdgeMove()

    # need to draw movement of a basic state
    def MoveBasic(self):
      distance = self.GUI.distanceHelper()
      self.GUI.drawBasicMove(self.GUI.selected, distance)

    # need to draw movement of an orthogonal state
    def MoveOrthogonal(self):
      distance = self.GUI.distanceHelper()
      self.GUI.drawOrthogonalMove(self.GUI.selected, distance)

    # need to draw movement of a composite state
    def MoveComposite(self):
      distance = self.GUI.distanceHelper()
      self.GUI.drawCompositeMove(self.GUI.selected, distance)

    # need to draw movement of a history state
    def MoveHistory(self):
      distance = self.GUI.distanceHelper()
      self.GUI.drawHistoryMove(self.GUI.selected, distance)

    # need to draw movement of edges associated with the selected entity
    def MoveEdge(self):
      distance = self.GUI.distanceHelper()
      inedges = self.GUI.inEdgeListHelper(self.GUI.selected[0])
      outedges = self.GUI.outEdgeListHelper(self.GUI.selected[0])
      self.GUI.drawEdgeMove(distance, inedges, outedges)

    # need to draw containment for an entity
    def ContainEntity(self):
      result = self.GUI.typeHelper(self.GUI.toContainFor)
      # this can only occur for orthogonal or composite states
      if result == "orthogonal":
        self.OrthogonalContain()
      elif result == "composite":
        self.CompositeContain()

    # need to draw containment for an orthogonal state
    def OrthogonalContain(self):
      self.GUI.drawOrthogonalContain(self.GUI.toContainFor)

    # need to draw containment for a composite state
    def CompositeContain(self):
      self.GUI.drawCompositeContain(self.GUI.toContainFor)

    # need to delete an entity
    def DeleteEntity(self):
      result = self.GUI.typeHelper(self.GUI.toDelete)
      # send event to delete entity
      if result == "basic":
        self.BasicDelete()
      elif result == "orthogonal":
        self.OrthogonalDelete()
      elif result == "composite":
        self.CompositeDelete()
      elif result == "history":
        self.HistoryDelete()
      elif result == "edge":
        self.EdgeDelete()

    # need to draw deletion of a basic state
    def DeleteBasic(self):
      self.GUI.drawBasicDelete(self.GUI.toDelete)

    # need to draw deletion of an orthogonal state
    def DeleteOrthogonal(self):
      self.GUI.drawOrthogonalDelete(self.GUI.toDelete)

    # need to draw deletion of a composite state
    def DeleteComposite(self):
      self.GUI.drawCompositeDelete(self.GUI.toDelete)

    # need to draw deletion of a history state
    def DeleteHistory(self):
      self.GUI.drawHistoryDelete(self.GUI.toDelete)

    # need to draw deletion of an edge
    def DeleteEdge(self):
      self.GUI.drawEdgeDelete(self.GUI.toDelete)


class ScopedGUI_Dynamic:
  
    def __init__(self, controller):
      self.controller = controller
      self.statechart = ScopedStatechart()
      self.statechart.initModel()     
      self.controller.bindDynamic(self.statechart)
      self.statechart.event("start", self.controller)

#======================================================================#
# GUI Static class

class ScopedGUI_Static(Frame):

  def __init__(self, parent, controller):
    Frame.__init__(self, parent)

    self.parent = parent
    self.controller = controller
    self.controller.bindStatic(self)

    # list of all Entities on the canvas
    # an Entity is a tuple:
    # each tuple has a State or Edge and 
    # a list of GUI IDs corresponding to the drawing of the Entity
    self.entitylist = []
 
    # (x,y) of the last click
    self.lastClickLocation = None

    # (x,y) of the current mouse position
    self.currentMousePos = None

    # (x,y) of the previous mouse position
    self.prevMousePos = None

    # the entity which we have selected
    self.selected = None

    # the entity which we have to deselect
    self.toDeselect = None

    # the entity which we have to delete
    self.toDelete = None

    # the entity for which we have to check containment
    self.toContainFor = None

    # the Entity from which we have to draw an edge
    self.fromEntity = None
    # the Entity to which we have to draw an edge
    self.toEntity = None

    # the GUI ID of the line which shows where an arc will be drawn
    self.showlineID = None

    # if we are moving an entity we lock all inputs
    self.moveLock = False

    # if we are creating an edge we lock all inputs
    self.arrowLock = False

    # if we are ready for a new AnyMotion event
    self.moveDone = True

    # get image for the buttonsmenu
    self.buttonsMenuImage = PhotoImage(file="./buttonsmenu.gif")
        
    # setup UI
    self.createWidgets()
    
    # user closes the window
    parent.protocol("WM_DELETE_WINDOW", self.controller.window_close)

  # sets up the UI
  def createWidgets(self):

    self.pack()
    self.displayCanvas = Canvas(master=self,
                           takefocus=1,
                           width=CANVAS_W, height=CANVAS_H,
                           background="white")
  
    self.displayCanvas.pack(fill=BOTH, expand=1)
    self.displayCanvas.focus_set()
    
    self.buttonsMenu = self.displayCanvas.create_image(0,0,image=self.buttonsMenuImage,anchor="nw")
                                              
    self.stateButton = self.displayCanvas.create_rectangle(statX0, statY0,
                                                           statX1, statY1,
                                                           fill='')

    self.compositeButton = self.displayCanvas.create_rectangle(compX0, compY0,
                                                               compX1, compY1,
                                                               fill='')
                                                             
    self.orthogonalButton = self.displayCanvas.create_rectangle(orthX0, orthY0,
                                                                orthX1, orthY1,
                                                                fill='')
                                                           
    self.historyButton = self.displayCanvas.create_rectangle(histX0, histY0,
                                                             histX1, histY1,
                                                             fill='')

        
    self.displayCanvas.bind("<ButtonPress-1>", self.mouse1Click)
    self.displayCanvas.bind("<ButtonRelease-1>", self.mouse1Release)
    self.displayCanvas.bind("<Control-ButtonPress-1>", self.ctrlMouse1Click)
    self.displayCanvas.bind("<Control-ButtonPress-3>", self.ctrlMouse3Click)
    self.displayCanvas.bind("<Delete>", self.deletePressed)
    self.displayCanvas.bind("<Motion>", self.motion)


  ##############
  # User Input #
  ##############

  # user left clicks
  def mouse1Click(self, event):
    if self.arrowLock:
      # we are creating an edge
      X = self.displayCanvas.canvasx(event.x)
      Y = self.displayCanvas.canvasy(event.y)   
      self.lastClickLocation = (X,Y)

      # we will need this when we draw the edge
      self.toEntity = self.clickedOnEntity()

      # if the toEntity is not a state we must not draw an edge to it
      if self.toEntity != None and not(self.isState(self.toEntity)):
        self.toEntity = None

      # send event to the statechart
      self.controller.ButtonPress1()

    elif not(self.moveLock):
      # we are not creating an edge nor moving an entity
      X = self.displayCanvas.canvasx(event.x)
      Y = self.displayCanvas.canvasy(event.y)   
      self.lastClickLocation = (X,Y)

      # check if the user clicked a menubutton
      # if so, send Reset and X Button event
      clicked = self.menuButtonIsClicked()
      if clicked == "state":
        self.controller.Reset()
        self.controller.StateButton()
      elif clicked == "composite":
        self.controller.Reset()
        self.controller.CompositeButton()
      elif clicked == "orthogonal":
        self.controller.Reset()
        self.controller.OrthogonalButton()
      elif clicked == "history":
        self.controller.Reset()
        self.controller.HistoryButton()

      # user did not click on a menubutton
      else:
        # check whether an entity was clicked
        clickedEntity = self.clickedOnEntity()

        if clickedEntity == None:
          # user did not click on an entity -> send deselect event
          self.toDeselect = self.selected
          self.selected = None
          self.controller.Deselect()

        elif clickedEntity == self.selected:
          # user clicked the selected entity => this indicates the user wants to move the entity
          # we do not allow an edge to move by itself
          if not(isinstance(clickedEntity[0], DC_HyperEdge)):
            self.moveLock = True
            self.currentMousePos = self.lastClickLocation # smooths movement
            self.prevMousePos = self.lastClickLocation # smooths movement
            self.controller.Move()

        else:
          # user clicked anentity which isnt the selected entity
          # => deselect previous and select the new one
          self.toDeselect = self.selected
          self.selected = clickedEntity
          self.controller.Deselect()
          self.controller.Select()

  # user releases left click
  def mouse1Release(self, event):
    if self.moveLock:
      # this indicates the user has ended the moving of an entity
      self.moveLock = False
      self.controller.ButtonRelease1()
      self.controller.Contain() # an entity has moved, we might need to draw different containment

  # user ctrl+left clicks, 
  # if this click is on an entity, this indicates the user wants to draw an edge from this entity
  def ctrlMouse1Click(self, event):
    if not(self.arrowLock) and not(self.moveLock):
      # we are not creating an edge or moving
      # deselect current selection
      self.toDeselect = self.selected
      self.selected = None
      self.controller.Deselect()

      X = self.displayCanvas.canvasx(event.x)
      Y = self.displayCanvas.canvasy(event.y)   
      self.lastClickLocation = (X,Y)

      # we will need this when we draw the arc
      self.fromEntity = self.clickedOnEntity()

      # if user clicked on an entity which allows an edge coming from it (not an edge)
      if self.fromEntity != None and self.isState(self.fromEntity):
        # we will now create an edge => set the lock
        self.arrowLock = True
        self.currentMousePos = self.lastClickLocation # smooths movement
        self.controller.ControlButtonPress1()


  # user ctrl+right clicks, 
  # this will trigger creation of an entity
  def ctrlMouse3Click(self, event):
    if not(self.arrowLock) and not(self.moveLock):
      # we are not creating an edge or moving
      # deselect current selection
      self.toDeselect = self.selected
      self.selected = None
      self.controller.Deselect()

      X = self.displayCanvas.canvasx(event.x)
      Y = self.displayCanvas.canvasy(event.y)   
      self.lastClickLocation = (X,Y)

      self.controller.ControlButtonPress3()

  # delete key is pressed
  # this indicates the user wants to delete the currently selected entity
  def deletePressed(self, event):
    if not(self.arrowLock) and not(self.moveLock):
      # we are not creating an edge or moving
      self.toDeselect = self.selected
      self.toDelete = self.selected
      self.selected = None
      self.controller.Deselect()
      self.controller.Delete() 

  # mouse moves
  def motion(self, event):
    if self.arrowLock or self.moveLock and self.moveDone:
      # we are creating an arrow or moving an entity and the previous move is completed
      # do a new move
      self.moveDone = False

      X = self.displayCanvas.canvasx(event.x)
      Y = self.displayCanvas.canvasy(event.y) 

      self.prevMousePos = self.currentMousePos 
      self.currentMousePos = (X,Y)

      self.controller.AnyMotion()


  ###########
  # Helpers #
  ###########

  # checks if a given entity is a state or an edge
  def isState(self, entity):
    return not(isinstance(entity[0],DC_HyperEdge)) and not(isinstance(entity[0],DC_Contains))

  # checks whether a point is within a bounding box
  def isIn(self, point, box):
    (pointx,pointy) = point
    (x0,y0,x1,y1) = box
    if pointx >= x0 and pointx <= x1 and pointy >= y0 and pointy <= y1:
      return True
    else:
      return False

  # checks whether the user clicked in one of the menu buttons
  def menuButtonIsClicked(self): 
    if self.isIn(self.lastClickLocation, (statX0,statY0,statX1,statY1)):
      # user clicked on the state button
      return "state"
    elif self.isIn(self.lastClickLocation, (compX0,compY0,compX1,compY1)):
      # user clicked on the composite button
      return "composite"
    elif self.isIn(self.lastClickLocation, (orthX0,orthY0,orthX1,orthY1)):
      # user clicked on the orthogonal button
      return "orthogonal"
    elif self.isIn(self.lastClickLocation, (histX0,histY0,histX1,histY1)):
      # user clicked on the history button
      return "history"
    else:
      # user didnt click on a button
      return ""      

  # checks whether a bounding box is fully within the drawing canvas
  def withinBounds(self, bounding_box):
    (X0,Y0,X1,Y1) = bounding_box
    if X0 >= 0 and X0 <= CANVAS_W and X1 >= 0 and X1 <= CANVAS_W:
      if Y0 > 100 and Y0 <= CANVAS_H and Y1 > 100 and Y1 <= CANVAS_H:
        return True
    return False
    
  # checks whether the given State name is unique
  def isUniqueName(self, name):
    for (state,IDs) in self.entitylist:
      if self.isState((state,IDs)) and name == state.name:
        return False

    return True

  # returns the entity which was clicked, None if no entity was clicked
  def clickedOnEntity(self):
    # get a tuple containing the GUI IDs of the clicked entities
    (X,Y) = self.lastClickLocation
    clickedIDs = self.displayCanvas.find_overlapping(X-1,Y-1,X+1,Y+1)

    if len(clickedIDs) == 0:
      # didnt click on an entity
      return None
    else:
      # select the first entity returned
      clickedID = clickedIDs[0]
      for (stateOrEdge,IDs) in self.entitylist:
        # if the clickedID is in the IDs of the entity
        if clickedID in IDs:
          return (stateOrEdge,IDs)


  # recursive
  # returns true if state1 contains state2, else it returns false
  def containsState(self, state1, state2):
    result = False

    for contain in state1.contains:
      containedState = contain.toState
      if containedState == state2:
        result = True
      elif isinstance(containedState,DC_Orthogonal) or isinstance(containedState,DC_Composite):
        # recursive call
        if self.containsState(containedState, state2):
          result = True

    return result

  # returns the entity corresponding to the state containing the input state, 
  # returns None if the input state is not contained
  def getParentEntity(self, state):
    edge = state.incontain
    if edge != None:
      return self.getEntity(edge.fromState)
    else:
      return None


  # helps for creating an edge
  # this returns either "transition", "contain", "failed" or "giveoptions"
  # transition means we have to draw a transition
  # contain means we have to draw a contain
  # failed means we could not draw the edge
  # giveoptions means we need to know whether the user wants
  #             the edge to be a contain or transition
  def createEdgeHelper(self):
    result = ""
    # we decide what to do based on the fromEntity and the toEntity
    (fromState, _) = self.fromEntity
  
    if self.toEntity != None:
      (toState, _) = self.toEntity 

      if isinstance(fromState, DC_Composite) and isinstance(toState, DC_Orthogonal):
        result = "contain"  
      elif isinstance(fromState, DC_Composite):
        result = "giveoptions"
      elif isinstance(toState, DC_Orthogonal):
        self.drawError("only a composite state can connect to an orthogonal state")
        result = "failed"
      elif isinstance(fromState, DC_Basic) or isinstance(fromState, DC_History):
        result = "transition"
      elif isinstance(fromState, DC_Orthogonal):
        result = "contain"  

    else:
      # user did not click an entity
      result = "failed"

    if result == "giveoptions" and fromState == toState:
      # we know we have to draw a transition because a state cannot contain itself
      result = "transition"
    elif result == "contain" and fromState == toState:
      self.drawError("a state cannot contain itself")
      result = "failed"
    elif result == "giveoptions":
      # check whether the toState already has something containing it
      # if so, it has to be a transition
      if toState.incontain != None:
        result = "transition"

      # check whether toState contains fromState
      # if so, it has to be a transition
      if isinstance(toState, DC_Orthogonal) or isinstance(toState, DC_Composite):
        if self.containsState(toState, fromState):
          result = "transition"

    elif result == "contain":
      # check whether the toState already has something containing it, if so -> failed
      if toState.incontain != None:
        self.drawError("this state already has something containing it")
        result = "failed"

      # check whether toState contains fromState, if so -> failed
      if isinstance(toState, DC_Orthogonal) or isinstance(toState, DC_Composite):
        if self.containsState(toState, fromState):
          self.drawError("a state cannot contain a state by which it is contained")
          result = "failed"
    
    # delete the showline
    self.displayCanvas.delete(self.showlineID)

    return result

  # returns the type of the state contained in the input entity
  # "basic", "orthogonal", "composite", "history", "edge" (or "") 
  def typeHelper(self, entity):
    if entity == None:
      return ""
    elif isinstance(entity[0], DC_Basic):
      return "basic"
    elif isinstance(entity[0], DC_Orthogonal):
      return "orthogonal"
    elif isinstance(entity[0], DC_Composite):
      return "composite"
    elif isinstance(entity[0], DC_History):
      return "history"
    elif isinstance(entity[0], DC_HyperEdge):
      return "edge"

  # recursive
  # returns a list of incoming edges going to all states contained by the input state
  # or to the input state itself
  def inEdgeListHelper(self, state):
    edgelist = []

    # add own incoming edges
    edgelist.extend(state.inedges)
    if state.incontain != None:
      edgelist.append(state.incontain)

    if isinstance(state, DC_Orthogonal) or isinstance(state, DC_Composite):
      for contains in state.contains:
        containedState = contains.toState
        # recursion
        edgelist.extend(self.inEdgeListHelper(containedState))

    return edgelist 

  # recursive
  # returns a list of outgoing edges coming from all states contained by the input state
  # or from the input state itself
  def outEdgeListHelper(self, state):
    edgelist = []
    # add own outgoing edges
    edgelist.extend(state.outedges)

    if isinstance(state, DC_Orthogonal) or isinstance(state, DC_Composite):
      # add own outgoing edges
      edgelist.extend(state.contains)

      for contains in state.contains:
        containedState = contains.toState
        # recursion
        edgelist.extend(self.outEdgeListHelper(containedState))

    return edgelist 

  # gets the left/right/up/down-most coordinate from all contained entities by the input state
  # when this is called we already know that all contained orthogonal/composite states will
  # have the outer coords of what they contain themselves
  # this means that recursion is not necessary
  def outerCoordsHelper(self, state):
    leftmost = CANVAS_W # intialize to max
    upmost = CANVAS_H   # intialize to max
    rightmost = 0       # intialize to min
    downmost = 0        # intialize to min

    for contain in state.contains:
      (state,IDs) = self.getEntity(contain.toState)
  
      (x0,y0,x1,y1) = self.displayCanvas.coords(IDs[0])
      if leftmost > x0: 
        leftmost = x0
      if upmost > y0: 
        upmost = y0
      if rightmost < x1: 
        rightmost = x1
      if downmost < y1: 
        downmost = y1
        
    return (leftmost, upmost, rightmost, downmost)

  # returns the graphical Entity belonging to a given state or edge
  # returns None if not found
  def getEntity(self, given):
    for (stateOrEdge,IDs) in self.entitylist:
      if stateOrEdge == given:
        return (stateOrEdge,IDs)

    return None
  
  # calculates how much entities need to move during entity movement
  def distanceHelper(self):
    (prevx,prevy) = self.prevMousePos
    (mousex,mousey) = self.currentMousePos

    # calculate the (x,y) distance between the mouse position and the previous mouse position
    x = mousex - prevx
    y = mousey - prevy

    return (x,y)

  ###########
  # Drawing #
  ###########

  # creates and draws the new DC_Basic state at lastClickLocation
  def drawBasicState(self):
    # set the bounding box
    # for a basic state this is the circle
    (x,y) = self.lastClickLocation
    bounding_box = (x-10,y-10,x+10,y+10)

    # check whether the bounding box is fully in the canvas
    if self.withinBounds(bounding_box):
      # list of UI IDs corresponding to this State
      IDs = []

      # create a unique name for the State
      i = 1
      name = "Basic1"
      while (not (self.isUniqueName(name))):
        i += 1
        name = "Basic%d" % i
        
      # create the State
      new = DC_Basic(name)

      # draw the State
      newID = self.displayCanvas.create_oval(x-10,y-10,x+10,y+10, fill='grey',outline='blue', width=2)
      IDs.append(newID)
   
      # draw the name 
      nameID = self.displayCanvas.create_text(x,y+20, text = new.name)
      IDs.append(nameID)

      # add the State and its GUI IDs to the entitylist
      self.entitylist.append((new, IDs))
    else: 
      self.drawError("Creation failed: Entity out of canvas")

  # creates and draws the new orthogonal state at lastClickLocation
  def drawOrthogonalState(self):
    # set the bounding box
    # for an orthogonal state this is the rectangle
    (x,y) = self.lastClickLocation
    bounding_box = (x-15,y-15,x+15,y+15)

    # check whether the bounding box is fully in the canvas
    if self.withinBounds(bounding_box):
      # list of UI IDs corresponding to this State
      IDs = []

      # create a unique name for the State
      i = 1
      name = "Orthogonal1"
      while (not (self.isUniqueName(name))):
        i += 1
        name = "Orthogonal%d" % i
        
      # create the State
      new = DC_Orthogonal(name)

      # draw the State
      newID = self.displayCanvas.create_rectangle(x-15,y-15,x+15,y+15, fill='',outline='grey', width=2)
      IDs.append(newID)

      # draw the name 
      nameID = self.displayCanvas.create_text(x-15,y-30, text=new.name, anchor='nw')
      IDs.append(nameID)

      # add the State and its GUI IDs to the entitylist
      self.entitylist.append((new, IDs))
    else: 
      self.drawError("Creation failed: Entity out of canvas")



  # creates and draws the new composite state at lastClickLocation
  def drawCompositeState(self):
    # set the bounding box
    # for a composite state this is the rectangle
    (x,y) = self.lastClickLocation
    bounding_box = (x-15,y-15,x+15,y+15)

    # check whether the bounding box is fully in the canvas
    if self.withinBounds(bounding_box):
      # list of UI IDs corresponding to this State
      IDs = []

      # create a unique name for the State
      i = 1
      name = "Composite1"
      while (not (self.isUniqueName(name))):
        i += 1
        name = "Composite%d" % i
        
      # create the State
      new = DC_Composite(name)

      # draw the State
      newID = self.displayCanvas.create_rectangle(x-15,y-15,x+15,y+15, fill='',outline='blue', width=2)
      IDs.append(newID)

      # draw the name 
      nameID = self.displayCanvas.create_text(x-15,y-30, text=new.name, anchor='nw')
      IDs.append(nameID)

      # add the State and its GUI IDs to the entitylist
      self.entitylist.append((new, IDs))
    else: 
      self.drawError("Creation failed: Entity out of canvas")

  # creates and draws the new history state at lastClickLocation
  def drawHistoryState(self):
    # set the bounding box
    # for a basic state this is the circle
    (x,y) = self.lastClickLocation
    bounding_box = (x-10,y-10,x+10,y+10)

    # check whether the bounding box is fully in the canvas
    if self.withinBounds(bounding_box):
            # list of UI IDs corresponding to this State
      IDs = []

      # create a unique name for the State
      i = 1
      name = "History1"
      while (not (self.isUniqueName(name))):
        i += 1
        name = "History%d" % i
        
      # create the State
      new = DC_History(name)

      # draw the State
      newID = self.displayCanvas.create_oval(x-10,y-10,x+10,y+10, 
                                             fill='lightblue',outline='blue', width=2)
      textID = self.displayCanvas.create_text(x,y, text="H")
      IDs.append(newID)
      IDs.append(textID)
   
      # draw the name 
      nameID = self.displayCanvas.create_text(x,y+20, text=new.name)
      IDs.append(nameID)

      # add the State and its GUI IDs to the entitylist
      self.entitylist.append((new, IDs))
    else: 
      self.drawError("Creation failed: Entity out of canvas")

  # updates the showline from the last clicked entity to the current mouse position
  def drawShowLine(self):
    # we always draw lines from the middle of the righthandside of an entity
    # get the coordinates for this
    (_,IDs) = self.fromEntity
    (x1,y1,x2,y2) = self.displayCanvas.coords(IDs[0])
    x0 = x2
    y0 = y1 + (y2-y1) / 2
    
    # current mouse position coordinates
    (m0,m1) = self.currentMousePos

    # delete previous showline
    if self.showlineID != None:
      self.displayCanvas.delete(self.showlineID)

    # draw new showline
    self.showlineID = self.displayCanvas.create_line(x0,y0,m0,m1,arrow="last",width=2)

    # ready for new anymotion
    self.moveDone = True

  # draws a transition from fromEntity to toEntity
  def drawTransition(self):
    IDs = []
    # we always draw lines from the middle of the righthandside of an entity
    # get the coordinates for this
    (fromState,fromIDs) = self.fromEntity
    (fx0,fy0,fx1,fy1) = self.displayCanvas.coords(fromIDs[0])
    x0 = fx1
    y0 = fy0 + (fy1-fy0) / 2

    # we always draw lines to the middle of the lefthandside of an entity
    # get the coordinates for this
    (toState,toIDs) = self.toEntity
    (tx0,ty0,tx1,ty1) = self.displayCanvas.coords(toIDs[0])
    x1 = tx0
    y1 = ty0 + (ty1-ty0) / 2

    # we draw the text above the middle of the arrow
    textx = x0 + (x1-x0) / 2 - 15
    texty = y0 + (y1-y0) / 2 - 15

    # create transition
    new = DC_HyperEdge("trigger",fromState,toState)
    fromState.outedges.append(new)
    toState.inedges.append(new)   

    # draw the transition
    lineID = self.displayCanvas.create_line(x0,y0,x1,y1,arrow="last",width=2)
    textID = self.displayCanvas.create_text(textx,texty, text=new.name)
    IDs.append(lineID)
    IDs.append(textID)

    # add the edge and its GUI IDs to the list
    self.entitylist.append((new, IDs))

    # reset fromEntity and toEntity
    self.fromEntity = None
    self.toEntity = None

  # draws a containment edge from fromEntity to toEntity
  def drawContainment(self):
    IDs = []
    # we always draw containment from the middle of the top of an entity
    # get the coordinates for this
    (fromState,fromIDs) = self.fromEntity
    (fx0,fy0,fx1,fy1) = self.displayCanvas.coords(fromIDs[0])
    x0 = fx0 + (fx1-fx0) / 2
    y0 = fy0

    # we always draw containment to the middle of the top of an entity
    # get the coordinates for this
    (toState,toIDs) = self.toEntity
    (tx0,ty0,tx1,ty1) = self.displayCanvas.coords(toIDs[0])
    x1 = tx0 + (tx1-tx0) / 2
    y1 = ty0

    # create contain
    new = DC_Contains(fromState,toState)
    fromState.contains.append(new)
    toState.incontain = new   

    # draw the containment
    lineID = self.displayCanvas.create_line(x0,y0,x1,y1,arrow="last",dash=(4,4),fill='',width=1)
    IDs.append(lineID)

    # add the edge and its GUI IDs to the list
    self.entitylist.append((new, IDs))

    # we will need to do a contain for the fromEntity
    self.toContainFor = self.fromEntity

    # reset fromEntity and toEntity
    self.fromEntity = None
    self.toEntity = None

  # asks the user whether the edge coming from a composite state is supposed to be a transition or a contain
  # returns either transition or contain depending on the user's answer
  def drawEdgeOptions(self):
    response = tkMessageBox.askquestion("Composite state", 
                 "Is this a transition?\n(yes -> creates transition)\n(no  -> creates containment)")
    if response == "yes":
      return "transition"
    else:
      return "contain"

  # need to draw deselection of a basic State
  def drawBasicDeselect(self, entity):
    (state,IDs) = entity
    if state.is_default:
      self.displayCanvas.itemconfig(IDs[0], outline='green')
    else:
      self.displayCanvas.itemconfig(IDs[0], outline='blue')      


  # recursive
  # need to draw deselection of an orthogonal State
  # the outedgelist is a list of outgoing edges coming from all to be deselected states
  # if any state has an incoming edge which is in the outedgelist, we must deselect this edge
  def drawOrthogonalDeselect(self, entity, outedgelist):
    (state,IDs) = entity
    self.displayCanvas.itemconfig(IDs[0], outline='grey')

    # draw deselection of every contained state and edge
    for contains in state.contains:
      containedState = contains.toState
      if isinstance(containedState, DC_Basic):
        self.drawBasicDeselect(self.getEntity(containedState))

        # because it is a basic state we also check the inedges
        for inedge in containedState.inedges:
          if inedge in outedgelist:
            self.drawEdgeDeselect(self.getEntity(inedge))  
    
      elif isinstance(containedState, DC_History):
        self.drawHistoryDeselect(self.getEntity(containedState))

        # because it is a history state we also check the inedges
        for inedge in containedState.inedges:
          if inedge in outedgelist:
            self.drawEdgeDeselect(self.getEntity(inedge))  

      elif isinstance(containedState, DC_Orthogonal):
        self.drawOrthogonalDeselect(self.getEntity(containedState), outedgelist)
      elif isinstance(containedState, DC_Composite):
        self.drawCompositeDeselect(self.getEntity(containedState), outedgelist)



  # recursive
  # need to draw deselection of a composite State
  # the outedgelist is a list of outgoing edges coming from all to be deselected states
  # if any state has an incoming edge which is in the outedgelist, we must deselect this edge
  def drawCompositeDeselect(self, entity, outedgelist):
    (state,IDs) = entity
    if state.is_default:
      self.displayCanvas.itemconfig(IDs[0], outline='green')
    else:
      self.displayCanvas.itemconfig(IDs[0], outline='blue')

    # draw deselection of the incoming edges (these are edges to itself)
    for inedge in state.inedges:
      if inedge in outedgelist:
        self.drawEdgeDeselect(self.getEntity(inedge))

    # draw deselection of every contained state and edge
    for contains in state.contains:
      containedState = contains.toState
      if isinstance(containedState, DC_Basic):
        self.drawBasicDeselect(self.getEntity(containedState))

        # because it is a basic state we also check the inedges
        for inedge in containedState.inedges:
          if inedge in outedgelist:
            self.drawEdgeDeselect(self.getEntity(inedge))  
    
      elif isinstance(containedState, DC_History):
        self.drawHistoryDeselect(self.getEntity(containedState))

        # because it is a history state we also check the inedges
        for inedge in containedState.inedges:
          if inedge in outedgelist:
            self.drawEdgeDeselect(self.getEntity(inedge))  

      elif isinstance(containedState, DC_Orthogonal):
        self.drawOrthogonalDeselect(self.getEntity(containedState), outedgelist)
      elif isinstance(containedState, DC_Composite):
        self.drawCompositeDeselect(self.getEntity(containedState), outedgelist)


  # need to draw deselection of a history State
  def drawHistoryDeselect(self, entity):
    (state,IDs) = entity
    self.displayCanvas.itemconfig(IDs[0], outline='blue')  

  # need to draw deselection of an edge
  def drawEdgeDeselect(self, entity):
    (edge,IDs) = entity
    self.displayCanvas.itemconfig(IDs[0], fill='black')

  # need to draw selection of a basic State
  def drawBasicSelect(self, entity):
    (state,IDs) = entity
    self.displayCanvas.itemconfig(IDs[0], outline='red')

  # recursive
  # need to draw selection of an orthogonal State
  # the outedgelist is a list of outgoing edges coming from all to be selected states
  # if any state has an incoming edge which is in the outedgelist, we must select this edge
  def drawOrthogonalSelect(self, entity, outedgelist):
    (state,IDs) = entity
    self.displayCanvas.itemconfig(IDs[0], outline='red')

    # draw selection of every contained state and edge
    for contains in state.contains:
      containedState = contains.toState
      if isinstance(containedState, DC_Basic):
        self.drawBasicSelect(self.getEntity(containedState))

        # because it is a basic state we also check the inedges
        for inedge in containedState.inedges:
          if inedge in outedgelist:
            self.drawEdgeSelect(self.getEntity(inedge))  
    
      elif isinstance(containedState, DC_History):
        self.drawHistorySelect(self.getEntity(containedState))

        # because it is a history state we also check the inedges
        for inedge in containedState.inedges:
          if inedge in outedgelist:
            self.drawEdgeSelect(self.getEntity(inedge))  

      elif isinstance(containedState, DC_Orthogonal):
        self.drawOrthogonalSelect(self.getEntity(containedState), outedgelist)
      elif isinstance(containedState, DC_Composite):
        self.drawCompositeSelect(self.getEntity(containedState), outedgelist)


  # recursive
  # need to draw selection of a composite State
  # the outedgelist is a list of outgoing edges coming from all to be selected states
  # if any state has an incoming edge which is in the outedgelist, we must select this edge
  def drawCompositeSelect(self, entity, outedgelist):
    (state,IDs) = entity
    self.displayCanvas.itemconfig(IDs[0], outline='red')

    # draw selection of the incoming edges (these are edges to itself)
    for inedge in state.inedges:
      if inedge in outedgelist:
        self.drawEdgeSelect(self.getEntity(inedge))

    # draw selection of every contained state and edge
    for contains in state.contains:
      containedState = contains.toState
      if isinstance(containedState, DC_Basic):
        self.drawBasicSelect(self.getEntity(containedState))

        # because it is a basic state we also check the inedges
        for inedge in containedState.inedges:
          if inedge in outedgelist:
            self.drawEdgeSelect(self.getEntity(inedge))  
    
      elif isinstance(containedState, DC_History):
        self.drawHistorySelect(self.getEntity(containedState))

        # because it is a history state we also check the inedges
        for inedge in containedState.inedges:
          if inedge in outedgelist:
            self.drawEdgeSelect(self.getEntity(inedge))  

      elif isinstance(containedState, DC_Orthogonal):
        self.drawOrthogonalSelect(self.getEntity(containedState), outedgelist)
      elif isinstance(containedState, DC_Composite):
        self.drawCompositeSelect(self.getEntity(containedState), outedgelist)


  # need to draw selection of a history State
  def drawHistorySelect(self, entity):
    (state,IDs) = entity
    self.displayCanvas.itemconfig(IDs[0], outline='red')  

  # need to draw selection of an edge
  def drawEdgeSelect(self, entity):
    (edge,IDs) = entity
    self.displayCanvas.itemconfig(IDs[0], fill='red')

  # need to draw movement for edges (edges cannot move on their own)
  def drawEdgeMove(self, distance, inedges, outedges):
    (x,y) = distance
 
    # move the endpoint of all incoming edges
    for edge in inedges:
      (_,IDs) = self.getEntity(edge)
  
      # for the line we only need to move the endpoint
      # get old coords
      (prevx0,prevy0,prevx1, prevy1) = self.displayCanvas.coords(IDs[0])
          
      # set new coords       
      self.displayCanvas.coords(IDs[0], prevx0,prevy0,prevx1+x,prevy1+y)

      if not(isinstance(edge,DC_Contains)) and not(edge in outedges):
        # this makes it so the text is only moved once if the edge has to fully move
        textx = prevx0 + (prevx1+x-prevx0) / 2 - 15
        texty = prevy0 + (prevy1+y-prevy0) / 2 - 15
        self.displayCanvas.coords(IDs[1],textx,texty)

    # move the startpoint of all outgoing edges
    for edge in outedges:
      (_,IDs) = self.getEntity(edge)
  
      # for the line we only need to move the startpoint
      # get old coords
      (prevx0,prevy0,prevx1, prevy1) = self.displayCanvas.coords(IDs[0])
          
      # set new coords       
      self.displayCanvas.coords(IDs[0], prevx0+x,prevy0+y,prevx1,prevy1)

      if not(isinstance(edge,DC_Contains)):
        textx = prevx0 + (prevx1-prevx0+x) / 2 - 15
        texty = prevy0 + (prevy1-prevy0+y) / 2 - 15
        self.displayCanvas.coords(IDs[1],textx,texty)

    # an entity has moved so we might need to do a contain
    self.toContainFor = self.getParentEntity(self.selected[0])

    # ready for new anymotion (moving of edges always comes after moving of state)
    self.moveDone = True

  # need to draw movement over distance (x,y) of a basic state
  def drawBasicMove(self, entity, distance):
    (state,IDs) = entity
    (x,y) = distance

    # move all of the GUI entities
    for ID in IDs:
      self.displayCanvas.move(ID,x,y)

  # recursive
  # need to draw movement over distance (x,y) of an orthogonal state
  def drawOrthogonalMove(self, entity, distance):
    (state,IDs) = entity
    (x,y) = distance
   
    # move all of the GUI entities
    for ID in IDs:
      self.displayCanvas.move(ID,x,y)

    # draw movement of every contained state
    for contains in state.contains:
      containedState = contains.toState
      if isinstance(containedState, DC_Basic):
        self.drawBasicMove(self.getEntity(containedState), distance) 
      elif isinstance(containedState, DC_History):
        self.drawHistoryMove(self.getEntity(containedState), distance)
      elif isinstance(containedState, DC_Orthogonal):
        self.drawOrthogonalMove(self.getEntity(containedState), distance)
      elif isinstance(containedState, DC_Composite):
        self.drawCompositeMove(self.getEntity(containedState), distance)

  # recursive
  # need to draw movement over distance (x,y) of a composite state
  def drawCompositeMove(self, entity, distance):
    (state,IDs) = entity
    (x,y) = distance
   
    # move all of the GUI entities
    for ID in IDs:
      self.displayCanvas.move(ID,x,y)

    # draw movement of every contained state
    for contains in state.contains:
      containedState = contains.toState
      if isinstance(containedState, DC_Basic):
        self.drawBasicMove(self.getEntity(containedState), distance) 
      elif isinstance(containedState, DC_History):
        self.drawHistoryMove(self.getEntity(containedState), distance)
      elif isinstance(containedState, DC_Orthogonal):
        self.drawOrthogonalMove(self.getEntity(containedState), distance)
      elif isinstance(containedState, DC_Composite):
        self.drawCompositeMove(self.getEntity(containedState), distance)

  # need to draw movement over distance (x,y) of a history state
  def drawHistoryMove(self, entity, distance):
    (state,IDs) = entity
    (x,y) = distance

    # move all of the GUI entities
    for ID in IDs:
      self.displayCanvas.move(ID,x,y)

  # the root call to the recursive drawOrthogonalContain/drawCompositeContain has
  # called this function to move all inedges/outedges corresponding to the new coordinates
  # of the orthogonal/composite states
  def drawEdgeContain(self, inedgelist, outedgelist):
    # move the endpoint of all incoming edges
    for edge in inedgelist:
      (_,edgeIDs) = self.getEntity(edge)
      (_,stateIDs) = self.getEntity(edge.toState)
    
      # contains will be done in outedges
      # get old start coords of the edge
      (x0,y0,_,_) = self.displayCanvas.coords(edgeIDs[0])
  
      # get coords of toState
      (sx0,sy0,sx1,sy1) = self.displayCanvas.coords(stateIDs[0])

      if isinstance(edge, DC_HyperEdge):
        # we always draw transition arrows to the middle of the lefthandside of an entity
        x1 = sx0
        y1 = sy0 + (sy1-sy0) / 2
      else:
        # we always draw contain arrows to the middle of the top of an entity
        x1 = sx0 + (sx1-sx0) / 2
        y1 = sy0 
          
      # set new coords for the edge     
      self.displayCanvas.coords(edgeIDs[0], x0,y0,x1,y1)

      if not isinstance(edge, DC_Contains):
        # set new coords for the text on the edge
        # we draw text above the middle of the arrow
        textx = x0 + (x1-x0) / 2 - 15
        texty = y0 + (y1-y0) / 2 - 15
        self.displayCanvas.coords(edgeIDs[1],textx,texty)
      
    # move start points of outedgelist
    for edge in outedgelist:
      (_,edgeIDs) = self.getEntity(edge)
      (_,stateIDs) = self.getEntity(edge.fromState)

      # get old end coords of the edge
      (_,_,x1,y1) = self.displayCanvas.coords(edgeIDs[0])
  
      # get coords of fromState
      (sx0,sy0,sx1,sy1) = self.displayCanvas.coords(stateIDs[0])

      if isinstance(edge, DC_HyperEdge):
        # we always draw transition arrows from the middle of the righthandside of an entity
        x0 = sx1
        y0 = sy0 + (sy1-sy0) / 2
      else:
        # we always draw contain arrows from the middle of the top of an entity
        x0 = sx0 + (sx1-sx0) / 2
        y0 = sy0 
          
      # set new coords for the edge     
      self.displayCanvas.coords(edgeIDs[0], x0,y0,x1,y1)

      # set new coords for the text on the edge (if present)
      if isinstance(edge,DC_HyperEdge):
        # we draw text above the middle of the arrow
        textx = x0 + (x1-x0) / 2 - 15
        texty = y0 + (y1-y0) / 2 - 15
        self.displayCanvas.coords(edgeIDs[1],textx,texty)

  # recursive
  # need to draw containment for an orthogonal state
  def drawOrthogonalContain(self, entity):
    (state,IDs) = entity

    # gets the leftmost/rightmost/upmost/downmost coordinate from
    # all contained entities by state
    (x0,y0,x1,y1) = self.outerCoordsHelper(state)

    # add some space in between
    x0 -= 15
    y0 -= 20
    x1 += 20
    y1 += 20

    # set coords of rectangle and name
    self.displayCanvas.coords(IDs[0], x0,y0,x1,y1)
    self.displayCanvas.coords(IDs[1], x0,y0-15)

    # we keep lists of all inedges and outedges from a state we draw contain for
    # at the end of the rootcall of the recursion we then call drawEdgeContain
    # with these lists
    inedgelist = []
    outedgelist = []

    # recursive calls for parent
    parent = self.getParentEntity(entity[0])
    if parent != None and isinstance(parent[0], DC_Orthogonal):
      (inedgelist,outedgelist) = self.drawOrthogonalContain(parent)
    elif parent != None and isinstance(parent[0], DC_Composite):
      (inedgelist,outedgelist) = self.drawCompositeContain(parent)

    # add own in/outedges to the lists
    inedgelist.extend(state.inedges)
    if state.incontain != None:
      inedgelist.append(state.incontain)
    outedgelist.extend(state.outedges)
    outedgelist.extend(state.contains) 

    # if this is the rootcall of the recursion we do drawEdgeContain
    # else we give our in/outedgelist to the caller 
    if entity == self.toContainFor:
      self.drawEdgeContain(inedgelist,outedgelist)
      self.toContainFor = None
    else:
      return (inedgelist,outedgelist)

  # recursive
  # need to draw containment for a composite state
  def drawCompositeContain(self, entity):
    (state,IDs) = entity

    # gets the leftmost/rightmost/upmost/downmost coordinate from
    # all contained entities by state
    (x0,y0,x1,y1) = self.outerCoordsHelper(state)

    # add some space in between
    x0 -= 15
    y0 -= 20
    x1 += 20
    y1 += 20

    # set coords of rectangle and name
    self.displayCanvas.coords(IDs[0], x0,y0,x1,y1)
    self.displayCanvas.coords(IDs[1], x0,y0-15)

    # we keep lists of all inedges and outedges from a state we draw contain for
    # at the end of the rootcall of the recursion we then call drawEdgeContain
    # with these lists
    inedgelist = []
    outedgelist = []

    # recursive calls for parent
    parent = self.getParentEntity(entity[0])
    if parent != None and isinstance(parent[0], DC_Orthogonal):
      (inedgelist,outedgelist) = self.drawOrthogonalContain(parent)
    elif parent != None and isinstance(parent[0], DC_Composite):
      (inedgelist,outedgelist) = self.drawCompositeContain(parent)

    # add own in/outedges to the lists
    inedgelist.extend(state.inedges)
    if state.incontain != None:
      inedgelist.append(state.incontain)
    outedgelist.extend(state.outedges)
    outedgelist.extend(state.contains) 

    # if this is the rootcall of the recursion we do drawEdgeContain
    # else we give our in/outedgelist to the caller 
    if entity == self.toContainFor:
      self.drawEdgeContain(inedgelist,outedgelist)
      self.toContainFor = None
    else:
      return (inedgelist,outedgelist)

  # need to draw deletion of a basic state
  def drawBasicDelete(self, entity):
    (state,IDs) = entity

    # delete all of the GUI entities belonging to the state
    for ID in IDs:
      self.displayCanvas.delete(ID)

    # delete all of the incoming/outgoing edges
    edgesToDelete = []
    if state.incontain != None:
      edgesToDelete.append(state.incontain)

    edgesToDelete.extend(state.inedges)
    edgesToDelete.extend(state.outedges)
   
    uniquelist = set(edgesToDelete) # remove duplicates (edges to self)

    for edge in uniquelist:
      self.drawEdgeDelete(self.getEntity(edge))

    # delete the entity from the entitylist
    self.entitylist.remove(entity)

  # need to draw deletion of an orthogonal state
  def drawOrthogonalDelete(self, entity):
    (state,IDs) = entity

    # delete all of the GUI entities belonging to the state
    for ID in IDs:
      self.displayCanvas.delete(ID)

    # delete all of the incoming/outgoing edges
    edgesToDelete = []
    if state.incontain != None:
      edgesToDelete.append(state.incontain)

    edgesToDelete.extend(state.outedges)
    edgesToDelete.extend(state.contains)

    for edge in edgesToDelete:
      self.drawEdgeDelete(self.getEntity(edge))

    # delete the entity from the entitylist
    self.entitylist.remove(entity)

  # need to draw deletion of a composite state
  def drawCompositeDelete(self, entity):
    (state,IDs) = entity

    # delete all of the GUI entities belonging to the state
    for ID in IDs:
      self.displayCanvas.delete(ID)

    # delete all of the incoming/outgoing edges
    edgesToDelete = []
    if state.incontain != None:
      edgesToDelete.append(state.incontain)

    edgesToDelete.extend(state.inedges)
    edgesToDelete.extend(state.outedges)
    edgesToDelete.extend(state.contains)
   
    uniquelist = set(edgesToDelete) # remove duplicates (edges to self)

    for edge in uniquelist:
      self.drawEdgeDelete(self.getEntity(edge))

    # delete the entity from the entitylist
    self.entitylist.remove(entity)

  # need to draw deletion of a history state
  def drawHistoryDelete(self, entity):
    (state,IDs) = entity

    # delete all of the GUI entities belonging to the state
    for ID in IDs:
      self.displayCanvas.delete(ID)

    # delete all of the incoming/outgoing edges
    edgesToDelete = []
    if state.incontain != None:
      edgesToDelete.append(state.incontain)

    edgesToDelete.extend(state.inedges)
    edgesToDelete.extend(state.outedges)
   
    uniquelist = set(edgesToDelete) # remove duplicates (edges to self)

    for edge in uniquelist:
      self.drawEdgeDelete(self.getEntity(edge))

    # delete the entity from the entitylist
    self.entitylist.remove(entity)

  # need to draw deletion of an edge (HyperEdge or Contains)
  # not that Contains edges can only be deleted when the entity they are pointing to/from gets deleted
  def drawEdgeDelete(self, entity):
    (edge,IDs) = entity

    # delete all of the GUI entities belonging to the edge
    for ID in IDs:
      self.displayCanvas.delete(ID)

    if isinstance(edge, DC_HyperEdge):     
      # need to delete itself from the in/outedges of its to/fromState
      edge.fromState.outedges.remove(edge)
      edge.toState.inedges.remove(edge)

    else:
      # need to delete itself from the incontain of its toState and the contains of its fromState
      edge.fromState.contains.remove(edge)
      edge.toState.incontain = None

    # delete the entity from the entitylist
    self.entitylist.remove(entity)

  # shows an error to the user using the given input string
  def drawError(self, error):
    response = tkMessageBox.showerror("Error",error)


