
from copy import deepcopy
from ..util.infinity import INFINITY
from ..core.match_algo import HimesisMatcher
from ..core.himesis import Himesis, HConstants as HC
from rule_primitive import RulePrimitive
from messages import MatchSet, Match, TransformationException


class Matcher(RulePrimitive):
	'''
		Binds the source graph according to the pre-condition pattern.
	'''
	def __init__(self, condition, max=INFINITY):
		'''
			Binds the source graph according to the pre-condition pattern.
			@param condition: The pre-condition pattern.
			@param max: The maximum number of matches.
		'''
		super(Matcher, self).__init__()
		self.max = max
		self.condition = condition
	
	def __str__(self):
		s = super(Matcher, self).__str__()
		s = s.split(' ')
		s.insert(1, '[%s]' % self.condition.name)
		return reduce(lambda x, y: '%s %s' % (x,y), s)
	
	def packet_in(self, packet):
		self.exception = None
		self.is_success = False
		if self.condition[Himesis.Constants.GUID] in packet.match_sets:
			matchSet = packet.match_sets[self.condition[Himesis.Constants.GUID]]
		else:
			matchSet = MatchSet()
		
		# Find the matches
		try:
			i = 1
			if i <= self.max:
				for mapping in self._match(packet.graph, packet.global_pivots):
					# Convert the mapping to a Match object
					match = Match()
					match.from_mapping(mapping, packet.graph, self.condition)
					matchSet.matches.append(match)
					i += 1
					if i > self.max:
						# We don't need any more matches
						break
		except Exception, e:
			self.is_success = False
			self.exception = TransformationException(e)
			self.exception.packet = packet
			self.exception.transformation_unit = self
			return packet
            
		# Don't forget to add the match set to the packet
		#even if no matches were found
		if len(matchSet.matches) > 0:
			packet.match_sets[self.condition[Himesis.Constants.GUID]] = matchSet
		
		# Identify that this is the condition we are currently processing
		packet.current = self.condition[Himesis.Constants.GUID]
		
		# Success only if matches were found
		self.is_success = len(matchSet.matches) > 0
		return packet

	
	'''
		yield a match given a condition (e.g., a LHS), pivots and NAC(s)	

		1. verify that no unbound NAC can be successfully matched (if so, 
			regardless of the	pattern in self.condition, there can be no match)
		2. iterate through possible mappings of the pattern in self.condition... 
			if no bound NAC can be successfully matched given one such mapping, 
			yield it 
			
		NOTE: iteration through boundNACs is done starting from NACs with larger
				bridges (i.e., larger intersections with their LHS)
		NOTE: pred1 and succ1 are used to optimize matching... without them, the
	  			matcher will compute the predecessors/successors of the source graph
			  	many times '''	
	def _match(self, graph, pivots_) :
		pred1 	 = {}
		succ1 	 = {}
		boundNACs = []
		pivots 	 = deepcopy(pivots_)
		pivots.to_source_node_indices(graph)
		
		for NAC in self.condition.NACs :
			if NAC.bridge.vcount() > 0:
				boundNACs.append(NAC)
			
			else :
				nacMatcher = HimesisMatcher(source_graph=graph, pattern_graph=NAC)
				nacPivots  = pivots.to_mapping(graph, NAC)
				try :
					for mapping in nacMatcher.match_iter(context=nacPivots) :
						if NAC[HC.MT_CONSTRAINT](mapping, graph) :
							return
				except : raise
				finally : nacMatcher.reset_recursion_limit()
				pred1 = nacMatcher.pred1
				succ1 = nacMatcher.succ1
		boundNACs.sort(key=lambda n : len(n.bridge), reverse=True)
		

		lhsMatcher = HimesisMatcher(\
								source_graph=graph, 
								pattern_graph=self.condition, 
								pred1=pred1, 
								succ1=succ1)
		lhsPivots  = pivots.to_mapping(graph, self.condition)
		try :
			for lhsMapping in lhsMatcher.match_iter(context=lhsPivots) :
				#make the mapping into {...,LHSlabel:graphIndex,...}
				match = Match()
				match.from_mapping(lhsMapping, graph, self.condition)

				if self.condition[HC.MT_CONSTRAINT](match.to_label_mapping(graph), graph):
					if len(boundNACs) == 0 :
						yield lhsMapping			

					else :				
						isLhsMappingValid = True
						for NAC in boundNACs :
							#make the mapping compatible with the NAC
							match = Match()
							match.from_mapping(lhsMapping, graph, self.condition)
							context = match.to_mapping(graph, NAC)
									
							nacMatcher = HimesisMatcher(\
													source_graph=graph, 
													pattern_graph=NAC, 
													pred1=pred1, 
													succ1=succ1)
							for nacMapping in nacMatcher.match_iter(context=context):
								#make the mapping into {...,NAClabel:graphIndex,...}
								match = Match()
								match.from_mapping(nacMapping, graph, NAC)

								if NAC[HC.MT_CONSTRAINT](match.to_label_mapping(graph), graph):
									isLhsMappingValid = False
									break
							if not isLhsMappingValid :
								break
						else:
							yield lhsMapping
		except :	raise
		finally : lhsMatcher.reset_recursion_limit()


