
from ..util.infinity import INFINITY
from srule import SRule
from ..tcore.rollbacker import Rollbacker


class XSRule(SRule):
    '''
        Applies the transformation on one match.
    '''
    def __init__(self, LHS, RHS, ignore_resolver=False, external_matches_only=False,
                 custom_resolution=lambda packet: False, max_iterations=INFINITY):
        '''
            Applies the transformation on all matches found with roll-back capability.
            @param LHS: The pre-condition pattern (LHS + NACs).
            @param RHS: The post-condition pattern (RHS).
            @param ignore_resolver: Specifies whether or not a resolver is needed. 
            @param external_matches_only: Resolve conflicts ignoring the matches found in this FRule.
            @param custom_resolution: Override the default resolution function.
            @param max_iterations: The maximum number of times to match.
        '''
        super(XSRule, self).__init__(LHS, RHS, ignore_resolver, external_matches_only, custom_resolution, max_iterations)
        # max_iterations=1 because no all matches have been exhausted after first application
        self.B = Rollbacker(condition=LHS, max_iterations=1)
    
    def packet_in(self, packet):
        self.exception = None
        self.is_success = False
        # Checkpoint the original packet
        self.B.packet_in(packet)
        if not self.B.is_success:
            self.exception = self.B.exception
            return packet
        # Match
        packet = self.M.packet_in(packet)
        if not self.M.is_success:
            packet = self.B.restore()
            if self.M.exception:
                self.exception = self.M.exception
            elif self.B.exception:
                self.exception = self.B.exception
            return packet
        # Choose the first match
        packet = self.I.packet_in(packet)
        if not self.I.is_success:
            packet = self.B.restore()
            if self.I.exception:
                self.exception = self.I.exception
            elif self.B.exception:
                self.exception = self.B.exception
            return packet
        while True:
            # Rewrite
            packet = self.W.packet_in(packet)
            if not self.W.is_success:
                packet = self.B.restore()
                if self.W.exception:
                    self.exception = self.W.exception
                elif self.B.exception:
                    self.exception = self.B.exception
                return packet
            # Rule has been applied once, so it's a success anyway
            self.is_success = True
            if self.I.iterations == self.I.max_iterations:
                return packet
            # Re-Match
            packet = self.M.packet_in(packet)
            if not self.M.is_success:
                self.exception = self.M.exception
                return packet
            # Choose another match
            packet = self.I.next_in(packet)
            # No more iterations are left
            if not self.I.is_success:
                if self.I.exception:
                    packet = self.B.restore()
                    if self.B.exception:
                        self.exception = self.B.exception
                    self.exception = self.I.exception
                    self.is_success = False
                return packet
    
    def next_in(self, packet):
        # Only one roll-back
        self.exception = None
        self.is_success = False
        packet = self.B.next_in(packet)
        if not self.B.is_success:
            self.exception = self.B.exception
        return packet
